import React, { useState } from 'react';
import { useNavigate } from 'react-router-dom';
import { calculateAgenticFootprint } from '../../utils/calculationEngine';

const AgenticAICalculator = ({ onCalculationComplete }) => {
  const navigate = useNavigate();
  const [orchestratorTier, setOrchestratorTier] = useState('tier2');
  const [orchestratorCalls, setOrchestratorCalls] = useState(5);
  const [region, setRegion] = useState('us');
  const [subTasks, setSubTasks] = useState([
    { id: 1, type: 'cloud', category: 'text', tier: 'tier1', count: 1 }
  ]);
  const [nextId, setNextId] = useState(2);
  
  // Tier options for orchestrator
  const tierOptions = [
    { value: 'tier1', label: 'Tier 1: Lightweight / Simple Q&A' },
    { value: 'tier2', label: 'Tier 2: General-Purpose / Moderate Complexity' },
    { value: 'tier3', label: 'Tier 3: Advanced / Complex Reasoning' }
  ];
  
  // Add a new sub-task
  const addSubTask = () => {
    setSubTasks([
      ...subTasks,
      { id: nextId, type: 'cloud', category: 'text', tier: 'tier1', count: 1 }
    ]);
    setNextId(nextId + 1);
  };
  
  // Remove a sub-task
  const removeSubTask = (id) => {
    setSubTasks(subTasks.filter(task => task.id !== id));
  };
  
  // Update a sub-task
  const updateSubTask = (id, field, value) => {
    setSubTasks(subTasks.map(task => 
      task.id === id ? { ...task, [field]: value } : task
    ));
  };
  
  // Handle calculation
  const handleCalculate = (e) => {
    e.preventDefault();
    
    // Calculate footprint
    const results = calculateAgenticFootprint(
      orchestratorTier,
      orchestratorCalls,
      subTasks,
      region
    );
    
    // Add metadata to results
    const calculationResults = {
      type: 'agentic',
      orchestratorTier,
      orchestratorCalls,
      subTasks,
      region,
      results,
      timestamp: new Date().toISOString()
    };
    
    // Pass results to parent component
    onCalculationComplete(calculationResults);
    
    // Navigate to results page
    navigate('/results');
  };
  
  return (
    <div className="calculator-form">
      <h2>Agentic AI Footprint Calculator</h2>
      <p className="info-text">
        Estimate the environmental footprint of AI agents that orchestrate multiple AI models.
      </p>
      
      <form onSubmit={handleCalculate}>
        <div className="orchestrator-section content-area">
          <h3>Orchestrator Overhead</h3>
          <p className="info-text">
            Estimate the reasoning/planning calls the agent makes to its orchestrating LLM.
          </p>
          
          <div className="form-group">
            <label htmlFor="orchestratorTier">Orchestrator LLM Tier:</label>
            <select 
              id="orchestratorTier" 
              value={orchestratorTier} 
              onChange={(e) => setOrchestratorTier(e.target.value)}
            >
              {tierOptions.map(option => (
                <option key={option.value} value={option.value}>
                  {option.label}
                </option>
              ))}
            </select>
          </div>
          
          <div className="form-group">
            <label htmlFor="orchestratorCalls">Number of Reasoning/Planning Calls:</label>
            <input 
              type="number" 
              id="orchestratorCalls" 
              min="1" 
              value={orchestratorCalls} 
              onChange={(e) => setOrchestratorCalls(Math.max(1, parseInt(e.target.value) || 1))}
            />
          </div>
          
          <div className="form-group">
            <label htmlFor="region">Primary Compute Region:</label>
            <select 
              id="region" 
              value={region} 
              onChange={(e) => setRegion(e.target.value)}
            >
              <option value="us">United States</option>
              <option value="eu">Europe</option>
              <option value="asia">Asia</option>
              <option value="aus">Australia</option>
            </select>
          </div>
        </div>
        
        <div className="subtasks-section content-area">
          <h3>Discrete Sub-Tasks</h3>
          <p className="info-text">
            Add each distinct AI task performed by the agent.
          </p>
          
          {subTasks.map(task => (
            <div key={task.id} className="subtask-item">
              <div className="subtask-header">
                <h4>Sub-Task #{task.id}</h4>
                <button 
                  type="button"
                  className="button secondary"
                  onClick={() => removeSubTask(task.id)}
                >
                  Remove
                </button>
              </div>
              
              <div className="form-group">
                <label htmlFor={`type-${task.id}`}>Execution Type:</label>
                <select 
                  id={`type-${task.id}`} 
                  value={task.type} 
                  onChange={(e) => updateSubTask(task.id, 'type', e.target.value)}
                >
                  <option value="cloud">Cloud AI</option>
                  <option value="local">Local AI</option>
                </select>
              </div>
              
              <div className="form-group">
                <label htmlFor={`category-${task.id}`}>AI Category:</label>
                <select 
                  id={`category-${task.id}`} 
                  value={task.category} 
                  onChange={(e) => updateSubTask(task.id, 'category', e.target.value)}
                >
                  <option value="text">Text Generation</option>
                  <option value="image">Image Generation</option>
                  <option value="video">Video Generation</option>
                  <option value="audio">Audio Generation</option>
                  <option value="analytical">Analytical / Classification</option>
                </select>
              </div>
              
              <div className="form-group">
                <label htmlFor={`tier-${task.id}`}>Complexity Tier:</label>
                <select 
                  id={`tier-${task.id}`} 
                  value={task.tier} 
                  onChange={(e) => updateSubTask(task.id, 'tier', e.target.value)}
                >
                  <option value="tier1">Tier 1 (Basic/Lightweight)</option>
                  {(task.category === 'text' || task.category === 'image') && (
                    <option value="tier2">Tier 2 (Moderate/Complex)</option>
                  )}
                  {task.category === 'text' && (
                    <option value="tier3">Tier 3 (Advanced/Long-Form)</option>
                  )}
                </select>
              </div>
              
              <div className="form-group">
                <label htmlFor={`count-${task.id}`}>Number of Executions:</label>
                <input 
                  type="number" 
                  id={`count-${task.id}`} 
                  min="1" 
                  value={task.count} 
                  onChange={(e) => updateSubTask(task.id, 'count', Math.max(1, parseInt(e.target.value) || 1))}
                />
              </div>
            </div>
          ))}
          
          <button type="button" className="button" onClick={addSubTask}>
            Add Sub-Task
          </button>
        </div>
        
        <div className="form-actions">
          <button type="submit" className="button">Calculate Footprint</button>
        </div>
      </form>
      
      <div className="info-box">
        <h3>About this Calculator</h3>
        <p className="info-text">
          This calculator estimates the environmental footprint of agentic AI platforms by breaking down 
          their operation into orchestration overhead and discrete sub-tasks. The total footprint is the 
          sum of these components.
        </p>
        <p className="info-text">
          <strong>Note:</strong> All figures are estimates and actual footprints may vary based on 
          specific implementations, optimizations, and infrastructure.
        </p>
      </div>
    </div>
  );
};

export default AgenticAICalculator;
