import React, { useState } from 'react';
import { useNavigate } from 'react-router-dom';
import { calculateCloudFootprint } from '../../utils/calculationEngine';

const CloudAICalculator = ({ onCalculationComplete }) => {
  const navigate = useNavigate();
  const [category, setCategory] = useState('text');
  const [tier, setTier] = useState('tier1');
  const [model, setModel] = useState('');
  const [volume, setVolume] = useState(1);
  const [region, setRegion] = useState('us');
  
  // Model options based on category
  const modelOptions = {
    text: [
      { value: '', label: 'Select a model (optional)' },
      { value: 'gpt4o', label: 'GPT-4o', tier: 'tier2' },
      { value: 'gpt35', label: 'GPT-3.5', tier: 'tier1' },
      { value: 'claude3sonnet', label: 'Claude 3 Sonnet', tier: 'tier2' },
      { value: 'claude3haiku', label: 'Claude 3 Haiku', tier: 'tier1' },
      { value: 'claude3opus', label: 'Claude 3 Opus', tier: 'tier3' },
      { value: 'llama2', label: 'Llama 2', tier: 'tier1' },
      { value: 'llama3', label: 'Llama 3', tier: 'tier2' },
      { value: 'mistral7b', label: 'Mistral 7B', tier: 'tier1' },
      { value: 'mixtral', label: 'Mixtral', tier: 'tier2' }
    ],
    image: [
      { value: '', label: 'Select a model (optional)' },
      { value: 'dalle3', label: 'DALL-E 3', tier: 'tier1' },
      { value: 'sdxl', label: 'Stable Diffusion XL', tier: 'tier1' },
      { value: 'midjourney', label: 'Midjourney', tier: 'tier1' }
    ],
    video: [
      { value: '', label: 'Select a model (optional)' },
      { value: 'sora', label: 'Sora', tier: 'tier1' }
    ],
    audio: [
      { value: '', label: 'Select a model (optional)' },
      { value: 'suno', label: 'Suno', tier: 'tier1' }
    ],
    analytical: [
      { value: '', label: 'Select a model (optional)' },
      { value: 'distilbert', label: 'DistilBERT', tier: 'tier1' },
      { value: 'resnet50', label: 'ResNet-50', tier: 'tier1' }
    ]
  };
  
  // Tier options based on category
  const tierOptions = {
    text: [
      { value: 'tier1', label: 'Tier 1: Lightweight / Simple Q&A / Efficient Models' },
      { value: 'tier2', label: 'Tier 2: General-Purpose / Moderate Complexity' },
      { value: 'tier3', label: 'Tier 3: Advanced / Complex Reasoning / Long-Form' }
    ],
    image: [
      { value: 'tier1', label: 'Tier 1: Standard Image Generation' },
      { value: 'tier2', label: 'Tier 2: High-Detail / Complex Image Generation' }
    ],
    video: [
      { value: 'tier1', label: 'Tier 1: Short-Form / Standard Definition Video' }
    ],
    audio: [
      { value: 'tier1', label: 'Tier 1: Basic Audio / Standard TTS' }
    ],
    analytical: [
      { value: 'tier1', label: 'Tier 1: Standard Analytical / Classification Tasks' }
    ]
  };
  
  // Volume label based on category
  const volumeLabel = {
    text: 'Number of queries/interactions',
    image: 'Number of images',
    video: 'Total seconds of video',
    audio: 'Total minutes of audio',
    analytical: 'Number of items/queries'
  };
  
  // Handle model selection
  const handleModelChange = (e) => {
    const selectedModel = e.target.value;
    setModel(selectedModel);
    
    // Auto-select tier based on model if a model is selected
    if (selectedModel) {
      const modelData = modelOptions[category].find(m => m.value === selectedModel);
      if (modelData && modelData.tier) {
        setTier(modelData.tier);
      }
    }
  };
  
  // Handle category change
  const handleCategoryChange = (e) => {
    const newCategory = e.target.value;
    setCategory(newCategory);
    setModel('');
    setTier(tierOptions[newCategory][0].value); // Set to first tier of new category
  };
  
  // Handle calculation
  const handleCalculate = (e) => {
    e.preventDefault();
    
    // Calculate footprint
    const results = calculateCloudFootprint(category, tier, volume, region);
    
    // Add metadata to results
    const calculationResults = {
      type: 'cloud',
      category,
      tier,
      model: model || 'Generic model',
      volume,
      region,
      results,
      timestamp: new Date().toISOString()
    };
    
    // Pass results to parent component
    onCalculationComplete(calculationResults);
    
    // Navigate to results page
    navigate('/results');
  };
  
  return (
    <div className="calculator-form">
      <h2>Cloud AI Footprint Calculator</h2>
      <p className="info-text">
        Estimate the environmental footprint of cloud-based AI services.
      </p>
      
      <form onSubmit={handleCalculate}>
        <div className="form-group">
          <label htmlFor="category">AI Task Type:</label>
          <select 
            id="category" 
            value={category} 
            onChange={handleCategoryChange}
          >
            <option value="text">Text Generation</option>
            <option value="image">Image Generation</option>
            <option value="video">Video Generation</option>
            <option value="audio">Audio Generation</option>
            <option value="analytical">Analytical / Classification</option>
          </select>
        </div>
        
        <div className="form-group">
          <label htmlFor="model">AI Model (Optional):</label>
          <select 
            id="model" 
            value={model} 
            onChange={handleModelChange}
          >
            {modelOptions[category].map(option => (
              <option key={option.value} value={option.value}>
                {option.label}
              </option>
            ))}
          </select>
        </div>
        
        <div className="form-group">
          <label htmlFor="tier">AI Complexity Tier:</label>
          <select 
            id="tier" 
            value={tier} 
            onChange={(e) => setTier(e.target.value)}
          >
            {tierOptions[category].map(option => (
              <option key={option.value} value={option.value}>
                {option.label}
              </option>
            ))}
          </select>
        </div>
        
        <div className="form-group">
          <label htmlFor="volume">{volumeLabel[category]}:</label>
          <input 
            type="number" 
            id="volume" 
            min="1" 
            value={volume} 
            onChange={(e) => setVolume(Math.max(1, parseInt(e.target.value) || 1))}
          />
        </div>
        
        <div className="form-group">
          <label htmlFor="region">Primary Compute Region:</label>
          <select 
            id="region" 
            value={region} 
            onChange={(e) => setRegion(e.target.value)}
          >
            <option value="us">United States</option>
            <option value="eu">Europe</option>
            <option value="asia">Asia</option>
            <option value="aus">Australia</option>
          </select>
        </div>
        
        <div className="form-actions">
          <button type="submit" className="button">Calculate Footprint</button>
        </div>
      </form>
      
      <div className="info-box">
        <h3>About this Calculator</h3>
        <p className="info-text">
          This calculator estimates the operational environmental footprint of cloud-based AI services 
          based on the type of AI task, complexity tier, and usage volume. The estimates are derived 
          from publicly available research and benchmarks.
        </p>
        <p className="info-text">
          <strong>Note:</strong> All figures are estimates and actual footprints may vary based on 
          specific implementations, optimizations, and infrastructure.
        </p>
      </div>
    </div>
  );
};

export default CloudAICalculator;
