import React, { useState } from 'react';
import { useNavigate } from 'react-router-dom';
import { calculateLocalFootprint } from '../../utils/calculationEngine';

const LocalAICalculator = ({ onCalculationComplete }) => {
  const navigate = useNavigate();
  const [hardwareCategory, setHardwareCategory] = useState('high-end-gpu');
  const [taskType, setTaskType] = useState('llm');
  const [estimationLevel, setEstimationLevel] = useState('basic');
  const [duration, setDuration] = useState(1);
  const [outputs, setOutputs] = useState(10);
  const [measuredPower, setMeasuredPower] = useState(350);
  const [region, setRegion] = useState('us');
  
  // Hardware category options
  const hardwareOptions = [
    { value: 'high-end-gpu', label: 'High-End GPU Desktop (e.g., RTX 4080/4090)' },
    { value: 'mid-range-gpu', label: 'Mid-Range GPU Desktop (e.g., RTX 4060Ti/4070)' },
    { value: 'entry-gpu', label: 'Entry GPU Desktop / CPU-only' },
    { value: 'high-perf-laptop', label: 'High-Performance AI Laptop' },
    { value: 'standard-laptop', label: 'Standard Laptop (iGPU/NPU)' },
    { value: 'apple-m-series', label: 'Apple M-Series Laptop' }
  ];
  
  // Task type options
  const taskOptions = [
    { value: 'llm', label: 'Large Language Model (LLM)' },
    { value: 'image', label: 'Image Generation' }
  ];
  
  // Representative power draw based on hardware and task
  const powerProfiles = {
    'high-end-gpu': {
      'llm': { min: 350, max: 550, avg: 450 },
      'image': { min: 300, max: 500, avg: 400 }
    },
    'mid-range-gpu': {
      'llm': { min: 200, max: 350, avg: 275 },
      'image': { min: 180, max: 300, avg: 240 }
    },
    'entry-gpu': {
      'llm': { min: 100, max: 200, avg: 150 },
      'image': { min: 100, max: 180, avg: 140 }
    },
    'high-perf-laptop': {
      'llm': { min: 100, max: 200, avg: 150 },
      'image': { min: 90, max: 180, avg: 135 }
    },
    'standard-laptop': {
      'llm': { min: 20, max: 60, avg: 40 },
      'image': { min: 25, max: 70, avg: 47.5 }
    },
    'apple-m-series': {
      'llm': { min: 15, max: 45, avg: 30 },
      'image': { min: 20, max: 50, avg: 35 }
    }
  };
  
  // Performance rates (simplified for MVP)
  const performanceRates = {
    'high-end-gpu': {
      'llm': 30, // tokens per second
      'image': 0.2 // images per second
    },
    'mid-range-gpu': {
      'llm': 20,
      'image': 0.1
    },
    'entry-gpu': {
      'llm': 10,
      'image': 0.05
    },
    'high-perf-laptop': {
      'llm': 15,
      'image': 0.08
    },
    'standard-laptop': {
      'llm': 5,
      'image': 0.02
    },
    'apple-m-series': {
      'llm': 8,
      'image': 0.03
    }
  };
  
  // Calculate estimated duration based on outputs and performance rate
  const calculateDuration = () => {
    const rate = performanceRates[hardwareCategory][taskType];
    if (taskType === 'llm') {
      // Assuming average of 100 tokens per output for LLM
      return (outputs * 100) / rate / 3600; // Convert to hours
    } else {
      // For images
      return outputs / rate / 3600; // Convert to hours
    }
  };
  
  // Handle output change
  const handleOutputsChange = (e) => {
    const value = Math.max(1, parseInt(e.target.value) || 1);
    setOutputs(value);
    if (estimationLevel === 'basic') {
      setDuration(calculateDuration());
    }
  };
  
  // Handle hardware category change
  const handleHardwareCategoryChange = (e) => {
    const value = e.target.value;
    setHardwareCategory(value);
    if (estimationLevel === 'basic') {
      setDuration(calculateDuration());
    }
    
    // Update measured power for advanced estimation
    const powerProfile = powerProfiles[value][taskType];
    setMeasuredPower(powerProfile.avg);
  };
  
  // Handle task type change
  const handleTaskTypeChange = (e) => {
    const value = e.target.value;
    setTaskType(value);
    if (estimationLevel === 'basic') {
      setDuration(calculateDuration());
    }
    
    // Update measured power for advanced estimation
    const powerProfile = powerProfiles[hardwareCategory][value];
    setMeasuredPower(powerProfile.avg);
  };
  
  // Handle calculation
  const handleCalculate = (e) => {
    e.preventDefault();
    
    // Calculate footprint
    const results = calculateLocalFootprint(
      hardwareCategory, 
      taskType, 
      duration, 
      region, 
      estimationLevel === 'advanced' ? measuredPower : null
    );
    
    // Add metadata to results
    const calculationResults = {
      type: 'local',
      hardwareCategory,
      taskType,
      estimationLevel,
      duration,
      outputs: estimationLevel === 'basic' ? outputs : null,
      measuredPower: estimationLevel === 'advanced' ? measuredPower : null,
      region,
      results,
      timestamp: new Date().toISOString()
    };
    
    // Pass results to parent component
    onCalculationComplete(calculationResults);
    
    // Navigate to results page
    navigate('/results');
  };
  
  return (
    <div className="calculator-form">
      <h2>Local AI Footprint Calculator</h2>
      <p className="info-text">
        Estimate the environmental footprint of running AI models on your local hardware.
      </p>
      
      <form onSubmit={handleCalculate}>
        <div className="form-group">
          <label htmlFor="estimationLevel">Estimation Level:</label>
          <div className="radio-group">
            <label>
              <input 
                type="radio" 
                name="estimationLevel" 
                value="basic" 
                checked={estimationLevel === 'basic'} 
                onChange={() => setEstimationLevel('basic')}
              />
              Basic (Hardware Profile)
            </label>
            <label>
              <input 
                type="radio" 
                name="estimationLevel" 
                value="advanced" 
                checked={estimationLevel === 'advanced'} 
                onChange={() => setEstimationLevel('advanced')}
              />
              Advanced (Your Measurements)
            </label>
          </div>
        </div>
        
        {estimationLevel === 'basic' ? (
          <>
            <div className="form-group">
              <label htmlFor="hardwareCategory">Hardware Category:</label>
              <select 
                id="hardwareCategory" 
                value={hardwareCategory} 
                onChange={handleHardwareCategoryChange}
              >
                {hardwareOptions.map(option => (
                  <option key={option.value} value={option.value}>
                    {option.label}
                  </option>
                ))}
              </select>
            </div>
            
            <div className="form-group">
              <label htmlFor="taskType">AI Task Type:</label>
              <select 
                id="taskType" 
                value={taskType} 
                onChange={handleTaskTypeChange}
              >
                {taskOptions.map(option => (
                  <option key={option.value} value={option.value}>
                    {option.label}
                  </option>
                ))}
              </select>
            </div>
            
            <div className="form-group">
              <label htmlFor="outputs">Number of Outputs:</label>
              <input 
                type="number" 
                id="outputs" 
                min="1" 
                value={outputs} 
                onChange={handleOutputsChange}
              />
              <span className="helper-text">
                {taskType === 'llm' ? 'Queries/Responses' : 'Images'}
              </span>
            </div>
            
            <div className="form-group">
              <label>Estimated Task Duration:</label>
              <div className="info-display">
                {duration.toFixed(2)} hours
              </div>
            </div>
          </>
        ) : (
          <>
            <div className="form-group">
              <label htmlFor="measuredPower">Measured Average System Power (Watts):</label>
              <input 
                type="number" 
                id="measuredPower" 
                min="1" 
                value={measuredPower} 
                onChange={(e) => setMeasuredPower(Math.max(1, parseInt(e.target.value) || 1))}
              />
            </div>
            
            <div className="form-group">
              <label htmlFor="duration">Actual Task Duration (hours):</label>
              <input 
                type="number" 
                id="duration" 
                min="0.01" 
                step="0.01" 
                value={duration} 
                onChange={(e) => setDuration(Math.max(0.01, parseFloat(e.target.value) || 0.01))}
              />
            </div>
          </>
        )}
        
        <div className="form-group">
          <label htmlFor="region">Your Region/Country:</label>
          <select 
            id="region" 
            value={region} 
            onChange={(e) => setRegion(e.target.value)}
          >
            <option value="us">United States</option>
            <option value="eu">Europe</option>
            <option value="asia">Asia</option>
            <option value="aus">Australia</option>
          </select>
        </div>
        
        <div className="form-actions">
          <button type="submit" className="button">Calculate Footprint</button>
        </div>
      </form>
      
      <div className="info-box">
        <h3>About this Calculator</h3>
        <p className="info-text">
          This calculator estimates the environmental footprint of running AI models on your local hardware. 
          The basic estimation uses representative power profiles for common hardware configurations, 
          while the advanced estimation allows you to input your own measured power consumption.
        </p>
        <p className="info-text">
          <strong>Note:</strong> All figures are estimates and actual footprints may vary based on 
          specific hardware configurations, optimizations, and usage patterns.
        </p>
      </div>
    </div>
  );
};

export default LocalAICalculator;
