import React, { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import { formatValue, generateEquivalencies } from '../utils/calculationEngine';

const ResultsPage = ({ results }) => {
  const navigate = useNavigate();
  const [timeframe, setTimeframe] = useState('daily');
  const [bufferPercent, setBufferPercent] = useState(20);
  
  // Redirect to calculator if no results
  useEffect(() => {
    if (!results) {
      navigate('/calculator');
    }
  }, [results, navigate]);
  
  // If no results, show loading or return null
  if (!results) {
    return null;
  }
  
  // Extract base results
  const baseEnergy = results.results.energy.value;
  const baseWater = results.results.water.value;
  const baseCarbon = results.results.carbon.value;
  
  // Calculate multipliers for different timeframes
  const multipliers = {
    daily: 1,
    weekly: 7,
    monthly: 30,
    yearly: 365
  };
  
  // Calculate current results with timeframe and buffer
  const currentResults = {
    energy: baseEnergy * multipliers[timeframe] * (1 + bufferPercent / 100),
    water: baseWater * multipliers[timeframe] * (1 + bufferPercent / 100),
    carbon: baseCarbon * multipliers[timeframe] * (1 + bufferPercent / 100)
  };
  
  // Generate equivalencies
  const equivalencies = generateEquivalencies(
    currentResults.energy / 1000, // Convert to kWh for equivalencies
    currentResults.water / 1000, // Convert to L for equivalencies
    currentResults.carbon / 1000  // Convert to kg for equivalencies
  );
  
  // Format values for display
  const formattedResults = {
    energy: formatValue(currentResults.energy, results.results.energy.unit),
    water: formatValue(currentResults.water, results.results.water.unit),
    carbon: formatValue(currentResults.carbon, results.results.carbon.unit)
  };
  
  // Handle buffer change
  const handleBufferChange = (e) => {
    setBufferPercent(parseInt(e.target.value));
  };
  
  return (
    <div className="container">
      <div className="content-area">
        <h1>Your AI Footprint Results</h1>
        
        <div className="results-meta">
          <p className="info-text">
            <strong>Calculation Type:</strong> {results.type === 'cloud' ? 'Cloud AI' : 
                                              results.type === 'local' ? 'Local AI' : 'Agentic AI'}
          </p>
          {results.type === 'cloud' && (
            <>
              <p className="info-text"><strong>AI Category:</strong> {results.category}</p>
              <p className="info-text"><strong>Model/Tier:</strong> {results.model}</p>
              <p className="info-text"><strong>Volume:</strong> {results.volume} {
                results.category === 'text' ? 'queries/interactions' :
                results.category === 'image' ? 'images' :
                results.category === 'video' ? 'seconds of video' :
                results.category === 'audio' ? 'minutes of audio' : 'queries'
              }</p>
            </>
          )}
          {results.type === 'local' && (
            <>
              <p className="info-text"><strong>Hardware:</strong> {results.hardwareCategory}</p>
              <p className="info-text"><strong>Task Type:</strong> {results.taskType}</p>
              <p className="info-text"><strong>Duration:</strong> {results.duration} hours</p>
            </>
          )}
        </div>
        
        <div className="timeframe-selector">
          <button 
            className={timeframe === 'daily' ? 'button' : 'button secondary'}
            onClick={() => setTimeframe('daily')}
          >
            Daily
          </button>
          <button 
            className={timeframe === 'weekly' ? 'button' : 'button secondary'}
            onClick={() => setTimeframe('weekly')}
          >
            Weekly
          </button>
          <button 
            className={timeframe === 'monthly' ? 'button' : 'button secondary'}
            onClick={() => setTimeframe('monthly')}
          >
            Monthly
          </button>
          <button 
            className={timeframe === 'yearly' ? 'button' : 'button secondary'}
            onClick={() => setTimeframe('yearly')}
          >
            Yearly
          </button>
        </div>
        
        <div className="results-grid grid">
          <div className="result-card content-area">
            <h2>Energy</h2>
            <div className="result-value">{formattedResults.energy}</div>
            <p className="equivalency info-text">
              {equivalencies.energy.description} {equivalencies.energy.value} {equivalencies.energy.unit}
            </p>
          </div>
          
          <div className="result-card content-area">
            <h2>Water</h2>
            <div className="result-value">{formattedResults.water}</div>
            <p className="equivalency info-text">
              {equivalencies.water.description} {equivalencies.water.value} {equivalencies.water.unit}
            </p>
          </div>
          
          <div className="result-card content-area">
            <h2>Carbon</h2>
            <div className="result-value">{formattedResults.carbon}</div>
            <p className="equivalency info-text">
              {equivalencies.carbon.description} {equivalencies.carbon.value} {equivalencies.carbon.unit}
            </p>
          </div>
        </div>
        
        <div className="offsetting-section content-area">
          <h2>Offsetting Recommendation</h2>
          <p className="info-text">
            Based on your AI usage, we recommend purchasing:
          </p>
          <ul>
            <li className="info-text">{Math.ceil(currentResults.carbon / 1000)} kg of carbon credits</li>
            <li className="info-text">{Math.ceil(currentResults.water / 1000)} liters of water credits</li>
          </ul>
          <div className="buffer-control">
            <label htmlFor="buffer">Uncertainty Buffer: {bufferPercent}%</label>
            <input 
              type="range" 
              id="buffer" 
              min="0" 
              max="100" 
              value={bufferPercent}
              onChange={handleBufferChange}
            />
          </div>
          <a href="/offsetting" className="button">Learn About Offsetting</a>
        </div>
        
        <div className="disclaimer-box">
          <p className="info-text">
            <strong>Note:</strong> All calculations are estimates based on available research data. 
            Actual environmental impact may vary based on specific hardware, data centers, and usage patterns.
          </p>
        </div>
        
        <div className="form-actions">
          <button className="button" onClick={() => navigate('/calculator')}>
            New Calculation
          </button>
        </div>
      </div>
    </div>
  );
};

export default ResultsPage;
