// Cloud AI tier data
export const cloudAITiers = {
  text: {
    tier1: {
      name: "Tier 1: Lightweight / Simple Q&A / Efficient Models",
      description: "<500 total tokens, simple lookups, classification",
      examples: "Llama-3.2 1B, GPT-4.1 nano (short); text classification",
      energy: { min: 0.002, max: 0.2, unit: "Wh" },
      water: { min: 0.01, max: 2.0, unit: "mL" },
      confidence: { energy: "High", water: "Medium" }
    },
    tier2: {
      name: "Tier 2: General-Purpose / Moderate Complexity",
      description: "500 - 2000 total tokens, standard chatbot use, content drafting",
      examples: "GPT-4o (short/medium), Claude 3.7 Sonnet (short/medium)",
      energy: { min: 0.3, max: 3.0, unit: "Wh" },
      water: { min: 1.5, max: 15, unit: "mL" },
      confidence: { energy: "High", water: "High" }
    },
    tier3: {
      name: "Tier 3: Advanced / Complex Reasoning / Long-Form",
      description: ">2000 tokens (up to 100k+), in-depth analysis, code gen, reasoning models",
      examples: "OpenAI 'o3', GPT-4.5; Claude 3.7 Sonnet ET, Claude 3 Opus; DeepSeek-R1",
      energy: { min: 5, max: 40, unit: "Wh" },
      water: { min: 20, max: 300, unit: "mL" },
      confidence: { energy: "Medium", water: "Medium" }
    }
  },
  image: {
    tier1: {
      name: "Tier 1: Standard Image Generation",
      description: "~0.25-1MP (e.g., 512x512-1024x1024px), 20-50 diffusion steps",
      examples: "Stable Diffusion v1.5/XL 1.0 (H100); Midjourney, DALL-E 3 (standard)",
      energy: { min: 0.1, max: 1.0, unit: "Wh" },
      water: { min: 0.05, max: 0.5, unit: "mL" },
      confidence: { energy: "Medium", water: "Medium" }
    },
    tier2: {
      name: "Tier 2: High-Detail / Complex Image Generation / Upscaling",
      description: ">1-4MP, >50-100 diffusion steps, complex prompts, upscaling",
      examples: "SDXL (older hardware/high settings); professional print/art",
      energy: { min: 1.0, max: 12.0, unit: "Wh" },
      water: { min: 0.5, max: 40, unit: "mL" },
      confidence: { energy: "Medium-Low", water: "Medium-Low" }
    }
  },
  video: {
    tier1: {
      name: "Tier 1: Short-Form / Standard Definition / Efficient Video",
      description: "<15-30s, ~480p-720p, simpler animations",
      examples: "Amazon Nova Reel, OpenAI Sora (lower-end/shorter clips), Open-Sora (derived 480p)",
      energy: { min: 1.0, max: 5.0, unit: "Wh" },
      water: { min: 0.35, max: 2.0, unit: "mL" },
      confidence: { energy: "Low-Medium", water: "Low-Medium" }
    }
  },
  audio: {
    tier1: {
      name: "Tier 1: Basic Audio / Standard TTS / Simple Loops",
      description: "Standard quality TTS, short simple instrumental loops",
      examples: "Efficient TTS (Google Cloud TTS), AudioLDM (simple outputs)",
      energy: { min: 0.001, max: 0.05, unit: "Wh" },
      water: { min: 0.00035, max: 0.02, unit: "mL" },
      confidence: { energy: "Medium", water: "Medium" }
    }
  },
  analytical: {
    tier1: {
      name: "Tier 1: Standard Analytical / Classification Tasks",
      description: "Pattern identification, input classification, object detection",
      examples: "Text classification (DistilBERT), image classification (ResNet-50)",
      energy: { min: 0.002, max: 0.04, unit: "Wh" },
      water: { min: 0.0007, max: 0.016, unit: "mL" },
      confidence: { energy: "High", water: "Medium-Low" }
    }
  }
};

// Local AI hardware profiles
export const localAIProfiles = {
  "high-end-gpu": {
    name: "High-End GPU Desktop (e.g., RTX 4080/4090)",
    llm: { min: 350, max: 550, avg: 450, unit: "W" },
    image: { min: 300, max: 500, avg: 400, unit: "W" }
  },
  "mid-range-gpu": {
    name: "Mid-Range GPU Desktop (e.g., RTX 4060Ti/4070)",
    llm: { min: 200, max: 350, avg: 275, unit: "W" },
    image: { min: 180, max: 300, avg: 240, unit: "W" }
  },
  "entry-gpu": {
    name: "Entry GPU Desktop / CPU-only",
    llm: { min: 100, max: 200, avg: 150, unit: "W" },
    image: { min: 100, max: 180, avg: 140, unit: "W" }
  },
  "high-perf-laptop": {
    name: "High-Performance AI Laptop",
    llm: { min: 100, max: 200, avg: 150, unit: "W" },
    image: { min: 90, max: 180, avg: 135, unit: "W" }
  },
  "standard-laptop": {
    name: "Standard Laptop (iGPU/NPU)",
    llm: { min: 20, max: 60, avg: 40, unit: "W" },
    image: { min: 25, max: 70, avg: 47.5, unit: "W" }
  },
  "apple-m-series": {
    name: "Apple M-Series Laptop",
    llm: { min: 15, max: 45, avg: 30, unit: "W" },
    image: { min: 20, max: 50, avg: 35, unit: "W" }
  }
};

// Performance rates for local AI
export const performanceRates = {
  "high-end-gpu": {
    llm: 30, // tokens per second
    image: 0.2 // images per second
  },
  "mid-range-gpu": {
    llm: 20,
    image: 0.1
  },
  "entry-gpu": {
    llm: 10,
    image: 0.05
  },
  "high-perf-laptop": {
    llm: 15,
    image: 0.08
  },
  "standard-laptop": {
    llm: 5,
    image: 0.02
  },
  "apple-m-series": {
    llm: 8,
    image: 0.03
  }
};

// Grid carbon intensity by region (gCO2e/kWh)
export const gridIntensity = {
  us: 385, // US average
  eu: 231, // EU average
  asia: 577, // Asia average (heavily influenced by China and India)
  aus: 452  // Australia average
};

// Water usage effectiveness
export const waterUsage = {
  // On-site cooling (L/kWh_IT)
  site: 0.30,
  
  // Off-site electricity generation (L/kWh_grid)
  source: {
    us: 3.142,
    eu: 2.876,
    asia: 6.016,
    aus: 3.890
  }
};

// Power usage effectiveness
export const pue = 1.12; // Representative of efficient hyperscalers

// Calculation functions
export const calculateCloudFootprint = (category, tier, volume, region) => {
  const tierData = cloudAITiers[category][tier];
  
  // Use average of min and max for calculations
  const energyPerUnit = (tierData.energy.min + tierData.energy.max) / 2;
  const waterPerUnit = (tierData.water.min + tierData.water.max) / 2;
  
  // Calculate total energy
  const totalEnergy = energyPerUnit * volume; // Wh
  
  // Calculate direct water usage (on-site cooling)
  const waterDirect = (totalEnergy / 1000) * waterUsage.site * 1000; // mL
  
  // Calculate indirect water usage (off-site electricity generation)
  const waterIndirect = (totalEnergy / 1000 * pue) * waterUsage.source[region] * 1000; // mL
  
  // Calculate total water usage
  const totalWater = waterDirect + waterIndirect; // mL
  
  // Calculate carbon emissions
  const totalCarbon = (totalEnergy / 1000) * pue * gridIntensity[region]; // gCO2e
  
  return {
    energy: {
      value: totalEnergy,
      unit: "Wh"
    },
    water: {
      value: totalWater,
      unit: "mL"
    },
    carbon: {
      value: totalCarbon,
      unit: "gCO2e"
    }
  };
};

export const calculateLocalFootprint = (hardwareCategory, taskType, duration, region, measuredPower = null) => {
  // Determine power draw
  let power;
  if (measuredPower !== null) {
    power = measuredPower; // User-provided power measurement
  } else {
    const profile = localAIProfiles[hardwareCategory][taskType];
    power = profile.avg; // Use average power from profile
  }
  
  // Calculate total energy
  const totalEnergy = power * duration; // Wh
  
  // Calculate water usage (simplified for local - only indirect from grid)
  const totalWater = (totalEnergy / 1000) * waterUsage.source[region] * 1000; // mL
  
  // Calculate carbon emissions
  const totalCarbon = (totalEnergy / 1000) * gridIntensity[region]; // gCO2e
  
  return {
    energy: {
      value: totalEnergy,
      unit: "Wh"
    },
    water: {
      value: totalWater,
      unit: "mL"
    },
    carbon: {
      value: totalCarbon,
      unit: "gCO2e"
    }
  };
};

export const calculateAgenticFootprint = (orchestratorTier, orchestratorCalls, subTasks, region) => {
  // Calculate orchestrator footprint
  const orchestratorFootprint = calculateCloudFootprint('text', orchestratorTier, orchestratorCalls, region);
  
  // Initialize totals
  let totalEnergy = orchestratorFootprint.energy.value;
  let totalWater = orchestratorFootprint.water.value;
  let totalCarbon = orchestratorFootprint.carbon.value;
  
  // Calculate footprint for each sub-task
  subTasks.forEach(task => {
    let taskFootprint;
    
    if (task.type === 'cloud') {
      taskFootprint = calculateCloudFootprint(task.category, task.tier, task.count, region);
    } else { // local
      // Simplified for MVP - using average duration and power
      const duration = 0.1; // 6 minutes per task
      const hardwareCategory = 'high-end-gpu'; // Default to high-end
      taskFootprint = calculateLocalFootprint(hardwareCategory, task.category === 'text' ? 'llm' : 'image', duration, region);
    }
    
    // Add to totals
    totalEnergy += taskFootprint.energy.value;
    totalWater += taskFootprint.water.value;
    totalCarbon += taskFootprint.carbon.value;
  });
  
  return {
    energy: {
      value: totalEnergy,
      unit: "Wh"
    },
    water: {
      value: totalWater,
      unit: "mL"
    },
    carbon: {
      value: totalCarbon,
      unit: "gCO2e"
    }
  };
};

// Format values for display
export const formatValue = (value, unit) => {
  if (unit === "Wh" && value >= 1000) {
    return `${(value / 1000).toFixed(2)} kWh`;
  } else if (unit === "mL" && value >= 1000) {
    return `${(value / 1000).toFixed(2)} L`;
  } else if (unit === "gCO2e" && value >= 1000) {
    return `${(value / 1000).toFixed(2)} kgCO2e`;
  } else {
    return `${value.toFixed(2)} ${unit}`;
  }
};

// Generate equivalencies
export const generateEquivalencies = (energy, water, carbon) => {
  return {
    energy: {
      description: 'Equivalent to running a laptop for',
      value: Math.round(energy * 10),
      unit: 'hours'
    },
    water: {
      description: 'Equivalent to',
      value: Math.round(water / 60),
      unit: 'showers'
    },
    carbon: {
      description: 'Equivalent to driving',
      value: Math.round(carbon / 1000 * 4),
      unit: 'miles in an average car'
    }
  };
};
